/* $Id: gmicp.c,v 1.6 1999/02/17 22:53:23 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* GM found a problem in the way we turn off ICP mode on E1433
   modules.  This tests that the problem is gone.  You must use an ICP
   Break-out Box, and use the -x command-line option to tell the
   program what DC level to look for on each channel, in order to
   really tell whether the problem is gone.  For the "stdtest" setup,
   you can use:

	./gmicp -x 0,0,3,0,3,0,0,3

   In this case, you should get an error from channel 4 if the bug is
   still there. */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	1024
#define	RANGE		2	/* Be careful about changing this! */
#define	SPAN		5000

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: gmicp.c,v 1.6 1999/02/17 22:53:23 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int *active)
{
    int     chan;

    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_input_mode(hw, group, E1432_INPUT_MODE_ICP));
    CHECK(e1432_set_range(hw, group, RANGE));
    CHECK(e1432_set_span(hw, group, SPAN));

    for (chan = 0; chan < nchan; chan++)
	active[chan] = 1;

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 4)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 2)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

/*ARGSUSED*/
static int
run_meas(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	 int *active, double *signal_list, int verbose)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    double  dc;
    int     scan, chan, i;

    /* We may just have turned off the ICP current source.  The input
       may not be connected to anything, in which case could take some
       time for the voltage to decay back to zero. */

    (void) sleep(1);

    CHECK(e1432_init_measure(hw, group));

    for (scan = 0; scan < 4; scan++)
    {
	/* Wait for block available */
	if (wait_block_avail(hw, group, scan, verbose,
			     BLOCKSIZE, SPAN) < 0)
	    return -1;

	/* Read the data */
	for (chan = 0; chan < nchan; chan++)
	{
	    if (!active[chan])
		continue;

	    if (verbose > 3)
		(void) printf("Reading chan %d\n", chan);

	    CHECK(e1432_read_float64_data(hw, chan_list[chan],
					  E1432_TIME_DATA,
					  buffer,
					  BLOCKSIZE, &trailer,
					  &count));
	    if (count != BLOCKSIZE)
	    {
		(void) fprintf(stderr,
			       "%s: e1432_read_float64_data: "
			       "actual count was %ld\n",
			       progname, count);
		return -1;
	    }

	    if (check_trailer(&trailer, 51200, SPAN,
			      chan_list[chan], 0) < 0)
		return -1;

	    if (signal_list[chan] >= 0)
	    {
		/* Calculate DC level */
		dc = 0;
		for (i = 0; i < BLOCKSIZE; i++)
		    dc += buffer[i];
		dc /= BLOCKSIZE;

		/* Make the limit big enough to accomodate both E1432
		   and E1433 modules when overloaded. */
		if (fabs(dc - signal_list[chan]) > RANGE/2 + 0.25)
		{
		    (void) fprintf(stderr,
				   "%s: scan %d chan %d: DC %g, expected %g\n",
				   progname, scan, chan, dc,
				   signal_list[chan]);
		    return -1;
		}
	    }
	}
    }

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    int *active, double *signal_list, int verbose)
{
    SHORTSIZ16 mode;
    int     chan, i;
    double  save_signal_list[NCHAN_MAX];

    for (chan = 0; chan < nchan; chan++)
	save_signal_list[chan] = signal_list[chan];

    CHECK(run_meas(hw, group, nchan, chan_list, active,
		   signal_list, verbose));

    if (verbose > 0)
	(void) printf("Inactive, top down\n");
    for (chan = nchan - 1; chan > 0; chan--)
    {
	if (verbose > 1)
	    (void) printf("Chan %d\n", chan);
	CHECK(e1432_set_active(hw, chan_list[chan], E1432_CHANNEL_OFF));
	active[chan] = 0;
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, verbose));
    }

    if (verbose > 0)
	(void) printf("Active, bottup up\n");
    for (chan = 0; chan < nchan; chan++)
    {
	if (verbose > 1)
	    (void) printf("Chan %d\n", chan);
	CHECK(e1432_set_active(hw, chan_list[chan], E1432_CHANNEL_ON));
	active[chan] = 1;
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, verbose));
    }

    if (verbose > 0)
	(void) printf("Inactive, bottup up\n");
    for (chan = 0; chan < nchan - 1; chan ++)
    {
	if (verbose > 1)
	    (void) printf("Chan %d\n", chan);
	CHECK(e1432_set_active(hw, chan_list[chan], E1432_CHANNEL_OFF));
	active[chan] = 0;
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, verbose));
    }

    if (verbose > 0)
	(void) printf("Active, top down\n");
    for (chan = nchan - 1; chan >= 0; chan--)
    {
	if (verbose > 1)
	    (void) printf("Chan %d\n", chan);
	CHECK(e1432_set_active(hw, chan_list[chan], E1432_CHANNEL_ON));
	active[chan] = 1;
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, verbose));
    }

    if (verbose > 0)
	(void) printf("Volt, top down\n");
    for (chan = nchan - 1; chan > 0; chan--)
    {
	if (verbose > 1)
	    (void) printf("Chan %d\n", chan);
	CHECK(e1432_set_input_mode(hw, chan_list[chan],
				   E1432_INPUT_MODE_VOLT));
	for (i = 0; i < nchan; i++)
	{
	    CHECK(e1432_get_input_mode(hw, chan_list[i], &mode));
	    if (i == chan && mode != E1432_INPUT_MODE_VOLT)
	    {
		(void) fprintf(stderr,
			       "%s: chan %d: failed to switch to VOLT mode\n",
			       progname, chan);
		return -1;
	    }
	    if (mode == E1432_INPUT_MODE_ICP)
		signal_list[i] = save_signal_list[i];
	    else
		signal_list[i] = 0;
	}
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, verbose));
    }

    if (verbose > 0)
	(void) printf("ICP, bottup up\n");
    for (chan = 0; chan < nchan; chan++)
    {
	if (verbose > 1)
	    (void) printf("Chan %d\n", chan);
	CHECK(e1432_set_input_mode(hw, chan_list[chan],
				   E1432_INPUT_MODE_ICP));
	for (i = 0; i < nchan; i++)
	{
	    CHECK(e1432_get_input_mode(hw, chan_list[i], &mode));
	    if (i == chan && mode != E1432_INPUT_MODE_ICP)
	    {
		(void) fprintf(stderr,
			       "%s: chan %d: failed to switch to ICP mode\n",
			       progname, chan);
		return -1;
	    }
	    if (mode == E1432_INPUT_MODE_ICP)
		signal_list[i] = save_signal_list[i];
	    else
		signal_list[i] = 0;
	}
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, verbose));
    }

    if (verbose > 0)
	(void) printf("Volt, bottup up\n");
    for (chan = 0; chan < nchan - 1; chan ++)
    {
	if (verbose > 1)
	    (void) printf("Chan %d\n", chan);
	CHECK(e1432_set_input_mode(hw, chan_list[chan],
				   E1432_INPUT_MODE_VOLT));
	for (i = 0; i < nchan; i++)
	{
	    CHECK(e1432_get_input_mode(hw, chan_list[i], &mode));
	    if (i == chan && mode != E1432_INPUT_MODE_VOLT)
	    {
		(void) fprintf(stderr,
			       "%s: chan %d: failed to switch to VOLT mode\n",
			       progname, chan);
		return -1;
	    }
	    if (mode == E1432_INPUT_MODE_ICP)
		signal_list[i] = save_signal_list[i];
	    else
		signal_list[i] = 0;
	}
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, verbose));
    }

    if (verbose > 0)
	(void) printf("ICP, top down\n");
    for (chan = nchan - 1; chan >= 0; chan--)
    {
	if (verbose > 1)
	    (void) printf("Chan %d\n", chan);
	CHECK(e1432_set_input_mode(hw, chan_list[chan],
				   E1432_INPUT_MODE_ICP));
	for (i = 0; i < nchan; i++)
	{
	    CHECK(e1432_get_input_mode(hw, chan_list[i], &mode));
	    if (i == chan && mode != E1432_INPUT_MODE_ICP)
	    {
		(void) fprintf(stderr,
			       "%s: chan %d: failed to switch to ICP mode\n",
			       progname, chan);
		return -1;
	    }
	    if (mode == E1432_INPUT_MODE_ICP)
		signal_list[i] = save_signal_list[i];
	    else
		signal_list[i] = 0;
	}
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, verbose));
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t[-x lev[,lev]]\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n"
		   "\t-x: Specify expected *DC* signal levels\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    double  signal_list[NCHAN_MAX];
    char   *p;
    int     active[NCHAN_MAX];
    int     c, i, nmod, verbose, group, nchan, chan, offset, count;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;
    for (chan = 0; chan < NCHAN_MAX; chan++)
	signal_list[chan] = -1;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:n:N:uvVx:")) != -1)
	switch (c)
	{
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'x':
	    offset = 0;
	    for (chan = 0; chan < NCHAN_MAX; chan++)
	    {
		if (sscanf(optarg + offset, "%lg%n",
			   &signal_list[chan], &count) != 1)
		{
		    if (strlen(optarg + offset) > 0)
		    {
			(void) fprintf(stderr,
				       "%s: invalid expected signal level: '%s'\n",
				       progname, optarg);
			usage();
		    }
		    break;
		}
		offset += count;
		count = 0;
		(void) sscanf(optarg + offset, " ,%n", &count);
		offset += count;
	    }
	    break;
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, active) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, active, signal_list, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
